/*
 * $Id: gen_date.c,v 1.6 1994/02/06 23:46:19 mintha Exp $
 *
 * Set of general routines for use in C programs.
 * These routines provide conversion function for dates and times.
 *
 * $Log: gen_date.c,v $
 * Revision 1.6  1994/02/06  23:46:19  mintha
 * Got rid of unused vars
 *
 * Revision 1.5  1993/11/30  22:43:14  mintha
 * Changed spm_to_char & hms_to_char to overwrite every second call
 * add month_to_str and month_to_num functions
 *
 * Revision 1.4  1993/10/20  09:54:10  mintha
 * include gen_utils_x.h now
 *
 * Revision 1.3  1993/10/08  10:10:22  mintha
 * Added hms_to_char like spm_to_char
 *
 * Revision 1.2  1993/10/05  09:27:59  mintha
 * Added stdio
 *
 * Revision 1.1  1993/09/23  09:18:28  mintha
 * Initial revision
 *
 */

#include <stdio.h>
#include <string.h>
#include <time.h>
#include "gen_utils_x.h"

static char *mnames[] = 
{
  "January", "February", "March", "April", "May", 
  "June", "july", "August", "September",
  "October", "November", "December"
};

/*
 * spm_to_hms - convert the time in seconds past midnight to an integer
 *              in the form HHMMSS
 */

int 
spm_to_hms(int time_in)
{
  int time_out;
  int hrs, min, sec;
  
  hrs = time_in / 3600;
  min = (time_in % 3600) /60;
  sec = (time_in % 60);

  time_out = hrs * 10000 + min * 100 + sec;

  return time_out;
}

/*
 * hms_to_spm - Given an integer in format HHMMSS convert it to seconds past
 *              midnight.
 */

int
hms_to_spm(int time)
{
  int hrs, min, sec;
  int tot_secs;
  
  hrs = time / 10000;
  min = (time % 10000) / 100;
  sec = (time % 100);
  
  tot_secs = hrs * 3600 + min * 60 + sec;
  
  return tot_secs;
}

/*
 * spm_to_char - Convert time in seconds past midnight to a string in the
 *               form HHMMSS or HH:MM:SS (if colon TRUE).  Returns pointer
 *               to a static string, overwritten every second call.
 */

char *
spm_to_char(int time, boolean colon)
{
  static char time_str1[10], time_str2[10];
  static boolean first_str = TRUE;
  char *str_ptr;
  int hrs, min, sec;

  if(first_str)
    str_ptr = time_str1;
  else
    str_ptr = time_str2;

  first_str = (!first_str);
  
  hrs = time / 3600;
  min = (time % 3600) / 60;
  sec = (time % 60);
  
  if(colon == TRUE)
    sprintf(str_ptr, "%.2d:%.2d:%.2d", hrs, min, sec);
  else
    sprintf(str_ptr, "%.2d%.2d%.2d", hrs, min, sec);

  return str_ptr;
}

/*
 * hms_to_char - Convert time in interger form HHMMSS to a string in the 
 *               form HHMMSS or HH:MM:SS (if colon TRUE).  Returns pointer
 *               to a static string, overwritten each call.
 */

char *
hms_to_char(int time, boolean colon)
{
  static char time_str1[10], time_str2[10];
  static boolean first_str = TRUE;
  char *str_ptr;
  int hrs, min, sec;

  if(first_str)
    str_ptr = time_str1;
  else
    str_ptr = time_str2;

  first_str = (!first_str);
  
  hrs = time / 10000;
  min = (time % 10000) / 100;
  sec = (time % 100);
  
  if(colon == TRUE)
    sprintf(str_ptr, "%.2d:%.2d:%.2d", hrs, min, sec);
  else
    sprintf(str_ptr, "%.2d%.2d%.2d", hrs, min, sec);

  return str_ptr;
}

/*
 * curr_date - Return the current time and date in a 24 character string with
 *             one of the following formats specified by type:
 *             VERBOSE: Sun Sep 16 01:03:52 1973\0.  
 *             LONG:    01:03:52 1973/09/16\0
 *             DATE:    1973/09/16\0
 *             TIME:    01:03:52
 *             The string returned is static and overwritten each call.
 *             Note: Programs that use dates that have just 2 digits for
 *             year (i.e. 73) are probably going to break in a few years
 *             so lets not do that.
 *             Also 09/08/1973 is too ambiguous given U.S. vs. Rest of World
 *             so the form Year/Month/Day is encouraged (also good for sorting)
 */

char *
time_to_char(int type)
{
  time_t tm;
  struct tm *s_tm;
  static char str[STR_LEN];
  
  tm = time(NULL);
  s_tm = localtime(&tm);
  
  switch(type)
    {
    case VERBOSE:
      strftime(str, STR_LEN, "%a %b %d %T %Y" , s_tm);
      break;
    case LONG:
      strftime(str, STR_LEN, "%T %Y/%m/%d" , s_tm);
      break;
    case DATE:
      strftime(str, STR_LEN, "%Y/%m/%d" , s_tm);
      break;
    case TIME:
      strftime(str, STR_LEN, "%T" , s_tm);
      break;
    }
  
  return str;
}

/*
 * curr_time - Current time in integer form: HHMMSS
 */

int
time_to_hms(void)
{
  time_t tm;
  struct tm *s_tm;
  int rtn_time;
  
  tm = time(NULL);
  s_tm = localtime(&tm);

  rtn_time = s_tm->tm_hour * 10000 + s_tm->tm_min * 100 + s_tm->tm_sec;
  
  return rtn_time;
}

/*
 * month_to_num - Convert month string to numeric (1-12).  Converts either
 *                Abbreviated or long names (based on length) 
 *                case insensitive.  Returns -1 if an error.
 */

int
month_to_num(char *str)
{
  int ctr = 0;
  
  while(ctr < 12)
    {
      if(strlen(str) == 3 && strncmp_nc(mnames[ctr], str, 3) == 0)
	break;
      else if(strcmp_nc(mnames[ctr], str) == 0)
	break;
      ctr++;
    }
  
  if(ctr == 12)
    ctr = -1;
  else
    ctr++;
  
  return ctr;
}

/*
 * month_to_str - Convert numeric month to string format.  If abbrev is
 *                true then abbreviated 3 character string is returned.
 *                Strings are static and overwritten every second call.
 */
  
char *
month_to_str(int mnum, boolean abbrev)
{
  static char month1[20], month2[20];
  static boolean first_str = TRUE;
  char *month_ptr;

  if(first_str)
    month_ptr = month1;
  else
    month_ptr = month2;
  
  first_str = (!first_str);
  
  if(mnum < 1 || mnum > 12)
    return NULL;
  
  strcpy(month_ptr, mnames[mnum - 1]);

  if(abbrev)
    {
      month_ptr[3] = '\0';
      upper_case(month_ptr);
    }

  return month_ptr;
}

			   
