/*
 * Program mkisofs.c - generate iso9660 filesystem  based upon directory
 * tree on hard disk.

   Written by Eric Youngdale (1993).

   Copyright 1993 Yggdrasil Computing, Incorporated

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "mkisofs.h"

#ifdef linux
#include <getopt.h>
#endif

#include "iso9660.h"
#include <ctype.h>
#include <time.h>
#include <stdlib.h>
#include <sys/stat.h>

#ifndef VMS
#include <unistd.h>
#endif

#include "exclude.h"

struct directory * root = NULL;

static char version_string[] = "mkisofs v1.00";

FILE * discimage;
unsigned int next_extent = 0;
unsigned int last_extent = 0;
unsigned int path_table_size = 0;
unsigned int path_table[4] = {0,};
unsigned int path_blocks = 0;
struct iso_directory_record root_record;
int use_RockRidge = 0;
int verbose = 0;
int all_files  = 0;
int follow_links = 0;
int generate_tables = 0;
static int timezone_offset;
char * extension_record = NULL;
int extension_record_extent = 0;
static  int extension_record_size = 0;

char * preparer = NULL;
char * publisher = NULL;


char * path_table_l = NULL;
char * path_table_m = NULL;
int goof = 0;

void usage(){
	fprintf(stderr,"Usage:\n");
	fprintf(stderr,"mkisofs [-o outfile] [-R] [-V volid] [-v] [-a] [-T] [-p preparer] [-P publisher] [-x path -x path ...] path\n");
	exit(1);
}

int get_iso9660_timezone_offset(){
  struct tm gm;
  struct tm * pt;
  time_t ctime;
  int local_min, gmt_min;

  time(&ctime);
  pt = gmtime(&ctime);
  gm = *pt;
  pt = localtime(&ctime);

  if(gm.tm_year < pt->tm_year)
    gm.tm_yday = -1;

  if(gm.tm_year > pt->tm_year)
    pt->tm_yday = -1;

  gmt_min = gm.tm_min + 60*(gm.tm_hour + 24*gm.tm_yday);
  local_min = pt->tm_min + 60*(pt->tm_hour + 24*pt->tm_yday);
  return (gmt_min - local_min)/15;
}


/* Fill in date in the iso9660 format */
int FDECL2(iso9660_date,char *, result, time_t, ctime){
  struct tm *local;
  local = localtime(&ctime);
  result[0] = local->tm_year;
  result[1] = local->tm_mon + 1;
  result[2] = local->tm_mday;
  result[3] = local->tm_hour;
  result[4] = local->tm_min;
  result[5] = local->tm_sec;
  result[6] = timezone_offset;
  return 0;
}

int FDECL3(iso9660_file_length,const char*, name, struct directory_entry *, sresult, 
			int, dirflag){
  int seen_dot = 0;
  char * result;
  int priority = 32767;
  int tildes = 0;
  int ignore = 0;
  int extra = 0;
  int current_length = 0;
  int chars_after_dot = 0;
  int chars_before_dot = 0;
  const char * pnt;
  result = sresult->isorec.name;

  if(strcmp(name,".") == 0){
    if(result) *result = 0;
    return 1;
  };

  if(strcmp(name,"..") == 0){
    if(result) *result = 1;
    return 1;
  };

  pnt = name;
  while(*pnt){
    if(*pnt == '#') {priority = 1; pnt++; continue; };
    if(*pnt == '~') {priority = 1; tildes++; pnt++; continue;};
    if(ignore) {pnt++; continue;};
    if(*pnt == '.') {
      if (seen_dot) {ignore++; continue;}
      if(result) *result++ = '.';
      seen_dot++;
    } else if (seen_dot) {
      if(chars_after_dot < 3) {
	chars_after_dot++;
	if(result) *result++ = toupper(*pnt);
      }
    } else {
      if(chars_before_dot < 8) {
	chars_before_dot++;
	if(result) *result++ = toupper(*pnt);
      };
    };
    current_length++;
    pnt++;
  };
  
  if(tildes == 2){
    int prio1 = 0;
    pnt = name;
    while (*pnt && *pnt != '~') pnt++;
    if (*pnt) pnt++;
    while(*pnt && *pnt != '~'){
      prio1 = 10*prio1 + *pnt - '0';
      pnt++;
    };
    priority = prio1;
  };
    
  if (!dirflag){
    if (!seen_dot) {
      if (result) *result++ = '.'; 
      extra++;
    };
    if(result){
      *result++ = ';';
      *result++ = '1';
    };
    extra += 2;
  };
		    
  if(result) *result++ = 0;
  sresult->priority = priority;
  return chars_before_dot + chars_after_dot + seen_dot + extra;
}

int FDECL2(main, int, argc, char **, argv){
  char * outfile, *volid;
  struct directory_entry de;
  unsigned int mem_start;
  struct stat statbuf;
  char * scan_tree;
  int c;

  if (argc < 2)
    usage();

  outfile = NULL;
  volid = "CDROM";
  while ((c = getopt(argc, argv, "o:V:RfvaTp:P:x:")) != EOF)
    switch (c)
      {
      case 'p':
	preparer = optarg;
	if(strlen(preparer) > 128) {
		fprintf(stderr,"Preparer string too long\n");
		exit(1);
	};
	break;
      case 'P':
	publisher = optarg;
	if(strlen(publisher) > 128) {
		fprintf(stderr,"Publisher string too long\n");
		exit(1);
	};
	break;
      case 'o':
	outfile = optarg;
	break;
      case 'f':
	follow_links++;
	break;
      case 'R':
	use_RockRidge++;
	break;
      case 'V':
	volid = optarg;
	break;
      case 'v':
	verbose++;
	break;
      case 'a':
	all_files++;
	break;
      case 'T':
	generate_tables++;
	break;
      case 'x':
        exclude(optarg);
	break;
      default:
	usage();
	exit(1);
      }

  mem_start = (unsigned int) sbrk(0);

  if(verbose) fprintf(stderr,"%s\n", version_string);
  /* Now find the timezone offset */

  timezone_offset = get_iso9660_timezone_offset();

  /*  The first step is to scan the directory tree, and take some notes */

  scan_tree = argv[optind];

  if(!scan_tree){
	  usage();
	  exit(1);
  };

  if(scan_tree[strlen(scan_tree)-1] != '/') {
    scan_tree = (char *) malloc(strlen(argv[optind])+2);
    strcpy(scan_tree, argv[optind]);
    strcat(scan_tree, "/");
  };

  if(use_RockRidge){
	extension_record = generate_rr_extension_record("RRIP_1991A",
				       "THE ROCK RIDGE INTERCHANGE PROTOCOL PROVIDES SUPPORT FOR POSIX FILE SYSTEM SEMANTICS",
				       "PLEASE CONTACT DISC PUBLISHER FOR SPECIFICATION SOURCE.  SEE PUBLISHER IDENTIFIER IN PRIMARY VOLUME DESCRIPTOR FOR CONTACT INFORMATION.", &extension_record_size);
  };

  stat(argv[optind], &statbuf);
  add_directory_hash(statbuf.st_dev, statbuf.st_ino);

  de.filedir = root;  /* We need this to bootstrap */
  scan_directory_tree(argv[optind], &de);
  root->self = root->contents;  /* Fix this up so that the path tables get done right */

  if(reloc_dir) sort_n_finish(reloc_dir);

  if (goof) exit(1);
  
  if (outfile){
	  discimage = fopen(outfile, "w");
	  if (!discimage){
		  fprintf(stderr,"Unable to open disc image file\n");
		  exit(1);

	  };
  } else
	  discimage =  stdout;

  /* Now assign addresses on the disc for the path table. */

  path_blocks = (path_table_size + (SECTOR_SIZE - 1)) >> 11;
  if (path_blocks & 1) path_blocks++;
  path_table[0] = 0x14;
  path_table[1] = path_table[0] + path_blocks;
  path_table[2] = path_table[1] + path_blocks;
  path_table[3] = path_table[2] + path_blocks;

  last_extent = path_table[3] + path_blocks;  /* The next free block */

  /* The next step is to go through the directory tree and assign extent
     numbers for all of the directories */

  assign_directory_addresses(root);

  if(extension_record) {
	  struct directory_entry * s_entry;
	  extension_record_extent = last_extent++;
	  s_entry = root->contents;
	  set_733(s_entry->rr_attributes + s_entry->rr_attr_size - 24,
		  extension_record_extent);
	  set_733(s_entry->rr_attributes + s_entry->rr_attr_size - 8,
		  extension_record_size);
  };

  if (use_RockRidge && reloc_dir)
	  finish_cl_pl_entries();

  /* Now we generate the path tables that are used by DOS to improve directory
     access times. */
  generate_path_tables();

  /* Generate root record for volume descriptor. */
  generate_root_record();

  dump_tree(root);

   iso_write(discimage, volid);

  fprintf(stderr,"Max brk space used %x\n", 
	  ((unsigned int)sbrk(0)) - mem_start);
  fprintf(stderr,"%d extents written (%d Mb)\n", last_extent, last_extent >> 9);
  return 0;
}

